/*
 *	LCD interface example
 *	Uses routines from delay.c
 *	This code will interface to a standard LCD controller
 *	like the Hitachi HD44780. It uses it in 4 bit mode, with
 *	the hardware connected as follows (the standard 14 pin 
 *	LCD connector is used):
 *	
 *	PORTD bits 0-3 are connected to the LCD data bits 4-7 (high nibble)
 *	PORTA bit 3 is connected to the LCD RS input (register select)
 *	PORTA bit 1 is connected to the LCD EN bit (enable)
 *	
 *	To use these routines, set up the port I/O (TRISA, TRISD) then
 *	call lcd_init(), then other routines as required.
 *	
 */

#include	<pic.h>
#include	"lcd.h"
#include	"delay.h"

#define	LCD_RS RC5
#define	LCD_RW RC6
#define LCD_EN RC7

#define LCD_DATA	PORTB

#define	LCD_STROBE()	((LCD_EN = 1),(LCD_EN=0))

void
lcd_data(unsigned char data)
{
	LCD_DATA = data << 4;
}

/* write a byte to the LCD in 4 bit mode */

void
lcd_write(unsigned char c)
{
	DelayUs(40);
	lcd_data( ( c >> 4 ) & 0x0F );
	LCD_STROBE();
	lcd_data( c & 0x0F );
	LCD_STROBE();
}

/*
 * 	Clear and home the LCD
 */

void
lcd_clear(void)
{
	LCD_RS = 0;
	lcd_write(0x1);
	DelayMs(2);
}

/* write a string of chars to the LCD */

void
lcd_puts(const char * s)
{
	LCD_RS = 1;	// write characters
	while(*s)
		lcd_write(*s++);
}

/* write one character to the LCD */

void
lcd_putch(char c)
{
	LCD_RS = 1;	// write characters
	lcd_write( c );
}


/*
 * Go to the specified position
 */

void
lcd_goto(unsigned char pos)
{
	LCD_RS = 0;
	lcd_write(0x80+pos);
}
	
/* initialise the LCD - put into 4 bit mode */
void
lcd_init()
{
	TRISC=0;
	TRISB=0;
	LCD_RS = 0;
	LCD_EN = 0;
	LCD_RW = 0;
	
	DelayMs(50);	// wait 15mSec after power applied,
	lcd_data(3);
	LCD_STROBE();
	DelayMs(10);
	lcd_data(3);
	LCD_STROBE();
	DelayMs(10);
	lcd_data(3);
	LCD_STROBE();
	DelayMs(10);
	lcd_data(2);	// Four bit mode
	LCD_STROBE();

	lcd_write(0x28); // Set interface length
	lcd_write(0b1100); // Display On, Cursor Off, Cursor Blink Off
	lcd_clear();	// Clear screen
	lcd_write(0x6); // Set entry Mode
}
