// Train Speed
// main.c
// 2007 admin@stillberg.com
// PIC16F690
// HI-TECH C Lite
#include <htc.h>
#include "lcd.h"
#include	"delay.h"
// _IESO_OFF 0x3BFF
// _FCMEN_OFF 0x37FF
__CONFIG(INTIO & WDTDIS & PWRTDIS & MCLRDIS & BORDIS & 0x3BFF & 0x37FF);

// sensor
#define RFR	!RA5
#define RFL	!RA4
// button
#define BTN1ON	!RA1
#define BTN2ON	!RA2
#define BTN3ON	!RA3
#define BTN1OFF	RA1
#define BTN2OFF	RA2
#define BTN3OFF	RA3
// led
#define LEDR	RC0
#define LEDL	RC1

unsigned long	count122 = 0;	// about 1/122 second count by prescale 32
unsigned char mode = 'S';
unsigned char dir = 'L';
unsigned char scale = 220;
unsigned long length = 111;	// mm
unsigned long passlen = 200;	// m

// service routine for timer 0 interrupt 
void interrupt
timer0_isr(void){
	count122++;
	T0IF = 0;
}

// long to string
void
ltoa(char *buf, int len, unsigned long val){
	int pos, v0;
	for( pos = 0; pos < len; pos++ ) {
		*(buf + pos) = ' ';
	}
	*(buf + len) = 0;
	for( pos = len - 1; pos >= 0 && val > 0; pos-- ) {
		v0 = val % 10;
		*(buf + pos) = '0' + v0;
		val = val / 10;
	}
}

// show mode, dir, scale, passlen on 1st line
void
showmode(void){
	char buf[17];

	lcd_goto(0);	// select first line
	switch( mode ) {
		case 'S': lcd_puts("STBY "); break;
		case 'W': lcd_puts("WAIT "); break;
		case 'T': lcd_puts("TIME "); break;
		case 'P': lcd_puts("PASS "); break;
	}
	switch( dir ) {
		case 'R': lcd_putch(0x7e); break;
		case 'L': lcd_putch(0x7f); break;
	}
	lcd_puts(" 1/");
	ltoa(buf, 3, scale);
	lcd_puts(buf);
	ltoa(buf, 4, passlen);
	lcd_puts(buf);
}

// show real and scaled speed on 2nd line
void
showspeed(long count){
	char buf[17];
	long rspeed, sspeed;

	rspeed = length * 122 / count;
	sspeed = length * scale * 4392 / 10000 / count; 
	lcd_goto(0x40);	// select second line
	ltoa(buf, 4, rspeed);
	lcd_puts(buf);
	lcd_puts("mm/s");
	ltoa(buf, 4, sspeed);
	lcd_puts(buf);
	lcd_puts("km/h");
}

// which button pressed 
char chkbtn(void) {
	long oncount;
	char result = 0;
	if( BTN1ON ) {
		while( 1 ) {
			if( BTN1OFF ) {
				DelayMs(20);
				if( BTN1OFF ) {
					break;
				}
			}
		}
		result = 1;
	} else if( BTN2ON ) {
		oncount = count122;
		while( 1 ) {
			if( BTN2OFF ) {
				DelayMs(20);
				if( BTN2OFF ) {
					break;
				}
			}
		}
		if( count122 > oncount + 60 ) { // over 0.5 sec pressed
			result = 4;
		} else {
			result = 2;
		}
	} else if( BTN3ON ) {
		while( 1 ) {
			if( BTN3OFF ) {
				DelayMs(20);
				if( BTN3OFF ) {
					break;
				}
			}
		}
		result = 3;
	}					
	return result;
}	

// change mode
void chmode(char m) {
	mode = m;
	showmode();
}

// change dir
void chdir(void) {
	dir = (dir == 'R') ? 'L' : 'R';
	showmode();
}

// change scale
void chscale(void) {
	scale = 
		scale == 80 ? 87 :
		scale == 87 ? 150 :
		scale == 150 ? 160 :
		scale == 160 ? 200 :
		scale == 200 ? 220 :
		scale == 220 ? 80 :
		220;
	showmode();
}

// change passlen
void chpasslen(void) {
	passlen += 100;
	if( passlen > 500 ) {
		passlen = 100;
	}
	showmode();
}

// save settings into EEPROM
void save(void) {
	unsigned char tmp;

	tmp = eeprom_read(0);
	if( dir != tmp ) {
		eeprom_write(0, dir);
	}
	tmp = eeprom_read(1);
	if( scale != tmp ) {
		eeprom_write(1, scale);
	}
	tmp = eeprom_read(2);
	if( passlen != tmp * 100 ) {
		eeprom_write(2, passlen / 100);
	}		
}

// load settings from EEPROM
void load(void) {
	unsigned char tmp;
	
	dir = eeprom_read(0);
	if( dir != 'R' && dir != 'L' ) {
		dir = 'L';
	}
	scale = eeprom_read(1);
	if( scale != 80 && scale != 87 && scale != 150 && scale != 160 &&
		scale != 200 && scale != 220 ) {
		scale = 220;
	}
	tmp = eeprom_read(2);
	if( tmp < 1 || tmp > 5 ) {
		tmp = 2;
	}
	passlen = tmp * 100;	
}

void main(void){
	char btn;
	long timedcount, passcount;
	char buf[17];

	lcd_init();
	
	OPTION = 0b0100;	// prescale by 32
	T0CS = 0;			// select internal clock
	T0IE = 1;			// enable timer interrupt
	GIE = 1;			// enable global interrupts
	
	ANSEL = 0x00;	// digital mode
	TRISA = 0xFF;	// input mode
	WPUA = 0x00;	// NO week pull up for port A

	load();	
	showmode();
	while (1){
		btn = chkbtn();
		if( mode == 'S' ) {	// STBY(stand by) mode
			if( btn == 1 ) {
				save();
				chmode('W');
			} else if( btn == 2 ) {
				chscale();
			} else if( btn == 3 ) {
				chdir();
			} else if( btn == 4 ) {
				chpasslen();
			}
			LEDR = RFR;
			LEDL = RFL;
		} else if( mode == 'W' ) { 	// WAIT mode
			if( btn == 1 ) {
				chmode('S');
			} else if( (dir == 'R' && RFL) || (dir == 'L' && RFR) ) {
				if( dir == 'R' ) {
					LEDL = 1;
				} else {
					LEDR = 1;
				}
				count122 = 0;
				chmode('T');
			}
		} else if( mode == 'T' ) {	// TIME mode
			if( btn == 1 ) {
				chmode('S');
			} else if( (dir == 'R' && RFR) || (dir == 'L' && RFL) ) {
				timedcount = count122;
				if( dir == 'R' ) {
					LEDR = 1;
				} else {
					LEDL = 1;
				}
				showspeed(timedcount);
				chmode('P');
				passcount = passlen * 1000 / length * timedcount / scale; 
				count122 = 0;
			}
		} else if( mode == 'P' ) {	// PASS mode
			if( btn == 1 ) {
				LEDR = 0;
				LEDL = 0;
				chmode('S');
			} else if( count122 > passcount ) {
				LEDR = 0;
				LEDL = 0;
				chmode('W');
			}
		}
	}
}
